/** @file   mobileobject.h
 * @brief   Declaration of MobileObject class.
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:42 $
 * @author  Tomi Lamminsaari
 */
 

#ifndef H_WWW_MOBILEOBJECT_H
#define H_WWW_MOBILEOBJECT_H

#include "vec2d.h"
#include <vector>


namespace WeWantWar {


/** @class  MobileObject
 * @brief   Provides the movement and collision services for child classes.
 * @author  Tomi Lamminsaari
 * @version $Revision: 1.1.1.1 $
 *
 * This class provides the functionality needed for the object that can
 * be placed and moved around the playfield.
 *
 * The collisions with other MobileObject s are implemented with bounding
 * spheres. If two objects are closer to each other than the sum of their
 * bounding spheres, they collide.
 *
 * The collisions with Map are examined with collisionpoints. Each
 * CollidableObject can have as many collisionpoints as they need. These
 * collisionpoints are relative to the position of this object. This
 * position information is maintained in our parent class @ref MobileObject
 *
 * Both of these collisiontypes can be enabled or disabled with collision
 * flags. By default, both collisions are enabled.
 */
class MobileObject
{
public:

  ///
  /// Datatypes and constants
  /// =======================
  
  /** @typedef  CollFlags
   * @brief     A type for bitflags that tell what kind of collisions
   *            are possible.
   */
  typedef unsigned int CollFlags;
  
  /** Object can't collide */
  static const CollFlags COLLIDE_NONE = 0;
  /** Object can collide with other objects. */
  static const CollFlags COLLIDE_OBJ = 1;
  /** Object can collide with map. */
  static const CollFlags COLLIDE_MAP = 2;



  ///
  /// Constructors and destructor.
  /// ============================
  
  /** Constructor. Places the object at coordinate (0,0).
   */
  MobileObject();
  
  
  /** Destructor
   */
  virtual ~MobileObject();
  

  
  ///
  /// Members
  /// =======
  
  /** Repositions this object to given coordinates.
   * @param     rP                New position
   */
  virtual void position( const eng2d::Vec2D& rP );
  
  /** Moves this object by given vector. If the COLLIDE_MAP - flag is on,
   * we check the map collisions and don't allow this object to go over
   * an obstackle mapblock.
   * @param     rMoveVec          Movement vector
   * @return    <code>true</code> if this object has collision with map
   *            in this new position.
   */
  virtual bool move( const eng2d::Vec2D& rMoveVec );
  
  /** Sets the rotation angle of this object.
   * @param     a                 New angle. 0...256
   */
  virtual void angle( int a );
  
  /** Changes the rotation angle by given amount.
   * @param     angleChange       Amount of angle change.
   */
  virtual void changeAngle( int angleChange );
  
  /** Sets the altitude.
   * @param     alt               New altitude. 0 means normal ground level
   */
  virtual void altitude( float alt );
  
  /** Sets the vertical speed.
   * @param     vertSpd           New vertical speed. Negative number lowers
   *                              this object, positive lifts it up.
   */
  virtual void verticalSpeed( float vertSpd );
  
  /** Sets the size of the bounding sphere.
   * @param     r                 Radius of the bounding sphere.
   */
  virtual void boundingSphere( float r );
  
  /** Adds new collisionpoint to this object. The collisionpoints are
   * relative to the position of this object.
   * @param     rP                The collisionpoint being added.
   */
  void addCollisionPoint( const eng2d::Vec2D& rP );
  
  /** Sets the existing collisionpoint.
   * @param     cindex            Index of the collisionpoint.
   * @param     rP                New data for that collisionpoint.
   */
  void setCollisionPoint( int cindex, const eng2d::Vec2D& rP );

  /** Sets the collision-flags.
   * @param     flags             New collisionflags or'd together.
   */
  void setCollisionFlags( CollFlags flags );
  
  /** Sets the force-vector. This vector is added to the current position
   * of this object. After that, the magnitude of this vector is decreased
   * so the force-effect fades away.
   * @param     rF                The force vector.
   */
  virtual void setForce( const eng2d::Vec2D& rF );

  /** Sets the <code>m_wallTouch</code> - member. The <code>move(...)</code>
   * - method sets this member to <code>true</code> each time the object
   * touches a wall during its movement.
   * @param     wtouch            : New wallTouch-value.
   */
  void setWallTouch( bool wtouch );
  
  
  
  ///
  /// Getter methods
  /// ==============
  
  /** Returns the current position of this object.
   * @return    Current position. In pixels from topleft corner of the
   *            playfield.
   */
  eng2d::Vec2D  position() const;
  
  /** Returns the current rotation angle.
   * @return    Rotation angle
   */
  int angle() const;
  
  /** Returns the current altitude.
   * @return    Current altitude.
   */
  float altitude() const;
  
  /** Returns the vertical speed.
   * @return    The vertical speed.
   */
  float verticalSpeed() const;
  
  /** Returns the radius of the bounding sphere.
   * @return    Radius of the bounding sphere
   */
  float boundingSphere() const;
  
  /** Returns the requested collisionpoint.
   * @param     cindex            Index of the collisionpoint. Shouldn't be
   *                              negative or greater than the number of
   *                              collisionpoints.
   * @return    The coordinates of the collisionpoint.
   */
  eng2d::Vec2D  getCollisionPoint( int cindex ) const;
  
  /** Returns the number of collisionpoints there are.
   * @return    Number of collisionpoints.
   */
  int getNumOfCollisionPoints() const;
  
  /** Tells if this object and given object collides with each other. Both
   * objects must have the COLLIDE_OBJ flag set to make the collision detection
   * work.
   * @param     pObj              Pointer to another collidable object.
   * @return    <code>true</code> if they collide.
   */
  bool collide( const MobileObject* pObj ) const;
  
  /** Tells if there is a collision with this object and given point. This
   * method does not consider the collision flags.
   * @param     rP                Coordinate of the point
   * @return    <code>true</code> if they collide.
   */
  bool collide( const eng2d::Vec2D& rP ) const;
  
  /** Tells if this object is currently colliding with the Map. The object
   * must have COLLIDE_MAP - flag set to make this work.
   * @return    <code>true</code> if there is a collision
   */
  bool hasMapCollision() const;
  
  /** Tells if this object would have collision with Map if it's translated
   * to the given coordinate.
   * @param     rP                The reference point.
   * @return    <code>true</code> if there would be a collision.
   */
  bool hasMapCollisionAt( const eng2d::Vec2D& rP ) const;
  
  /** Returns the collisionflags.
   * @return    The current collisionflags.
   */
  CollFlags getCollisionFlags() const;
  
  /** Returns the current force-vector.
   * @return    Current force-vector.
   */
  eng2d::Vec2D  getForce() const;
  
  /** Returns the <code>m_wallTouch</code> - member.
   * @return    Have we touched the wall since last reser.
   */
  bool getWallTouch() const;
  
  
protected:

  /** Updates the movements. We apply the force vector and vertical
   * speed.
   */
  virtual void updateMovement();
  

  ///
  /// Members
  /// =======
  
public:

  /** The current position of this object. */
  eng2d::Vec2D  m_position;
  
  /** The altitude of this object. 0 means normal ground level. */
  float   m_altitude;
  
  /** The vertical speed. */
  float   m_verticalSpeed;
  
  /** The rotation angle of this object. */
  int m_angle;
  
  /** The radius of the bounding sphere. */
  float m_boundingSphere;
  
  /** A vector that holds the collisionpoints. */
  std::vector< eng2d::Vec2D > m_collisionPoints;
  
  /** The collision flags. */
  CollFlags m_collisionFlags;
  
  /** The force-vector. */
  eng2d::Vec2D  m_force;
  
  /** A walltouch-member that keeps track on when we've touched the walls. */
  bool  m_wallTouch;
  
  
private:
  /** Disabled copycontructor.
   */
  MobileObject( const MobileObject& rMO );
  /** Disabled assignment operator
   */
  MobileObject& operator = ( const MobileObject& rMO );
};

};  // end of namespace

#endif // MOBILEOBJECT_H

/**
 * Version history
 * ===============
 * $Log: mobileobject.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:17:05+02  lamminsa
 * Initial revision
 *
 */
 
